#include "LedMatrixHelper.h"
#include "ChestsTrigonometryHelper.h"

#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>
#include <PubSubClient.h>
#include <ArduinoJson.h> 
#include <string.h>
#include <Servo.h>

Servo compassServo;


// ####### 1.a. CONFIGURATION ET DECLARATION DU HOSTPOT WIFI #######
char ssid[] = "";     //  le nom du reseau WIFI
char password[] = "";  // le mot de passe WIFI
WiFiClient espClient;

// ####### 1.b. DECLARATION DU CLIENT HTTP ET DES VARIABLES DE STOCKAGE DES DONNÉES DE JEU #######
HTTPClient http;
int teamPixelArt[8][8];
StaticJsonDocument<2048> requiredChestsDoc;
JsonArray requiredChests = requiredChestsDoc.to<JsonArray>();


// ####### 1.c. ACQUISITION DU PIXEL ART VIA REQUETE HTTP ET LECTURE DU JSON #######
void fetchTeamPixelArt() {
  http.useHTTP10(true);
  http.begin(espClient, "http://192.168.1.17:3000/pixel-arts/yellow");
  http.GET();
  DynamicJsonDocument doc(2048);
  deserializeJson(doc, http.getStream());  
  JsonArray teamPixelArtArray = doc["pixelArt"].as<JsonArray>();
  for (int i = 0; i < teamPixelArtArray.size(); i++) {
    for (int j = 0; j < teamPixelArtArray[0].size(); j++) {
      teamPixelArt[i][j] = teamPixelArtArray[i][j].as<int>();
      Serial.print(teamPixelArt[i][j]);
      Serial.print(" ");
    }
    Serial.println(); 
  } 
  http.end();
}

bool isColorInTeamPixelArt(int color) {
  for (int i = 0; i < 8; i++) {
    for (int j = 0; j < 8; j++) {
      if (teamPixelArt[i][j] == color) return true;         
    }
  }
  return false;
}

// ####### 1.d. ACQUISITION DES COFFRES COMPORTANT LES COULEURS DONT NOUS AVONS BESOIN #######
void fetchRequiredChests() {
  http.useHTTP10(true);
  http.begin(espClient, "http://192.168.1.17:3000/chests");
  http.GET();
  DynamicJsonDocument doc(4096);
  deserializeJson(doc, http.getStream());
  JsonArray chestArray = doc.as<JsonArray>();
  for (JsonObject chest : chestArray) {
    int chestContentColor = chest["content"].as<int>();
    if (isColorInTeamPixelArt(chestContentColor)) {
      requiredChests.add(chest);
    }
  }
  for (JsonObject c : requiredChests) {
    Serial.print("Coffre à chercher : \"");
    Serial.print(c["name"].as<String>());
    Serial.print("\" aux coordonnées: x:");
    Serial.print(c["location"]["x"].as<int>());
    Serial.print(" , y:");
    Serial.print(c["location"]["y"].as<int>());
    Serial.print(" , z:");
    Serial.println(c["location"]["z"].as<int>());
  }
}

// ####### 2.a. CONFIGURATION ET DECLARATION DU SERVEUR MQTT #######
char mqtt_server[] = "192.168.1.17";  //adresse IP serveur 
#define MQTT_USER ""
#define MQTT_PASS ""
PubSubClient MQTTclient(espClient);


//C'est cadeau pour vous éviter de vous taper du parsing à la main en C :p 
String getSubTopicAtIndex(String topic, int index) {
  int found = 0;
  int strIndex[] = {0, -1};
  int maxIndex = topic.length()-1;
  for(int i=0; i<=maxIndex && found<=index; i++){
    if(topic.charAt(i)=='/' || i==maxIndex){
        found++;
        strIndex[0] = strIndex[1]+1;
        strIndex[1] = (i == maxIndex) ? i+1 : i;
    }
  }
  return found>index ? topic.substring(strIndex[0], strIndex[1]) : "";
}

void MQTTconnect() {
  while (!MQTTclient.connected()) {
      Serial.print("Attente de la connexion MQTT....");
      String clientId = "TestClient-";
      clientId += String(random(0xffff), HEX);

    // test connexion
    if (MQTTclient.connect(clientId.c_str(),"","")) {
      Serial.println("connected");

      // ####### 2.a. SOUSCRIPTION AUX TOPICS MQTT DE MON/MES JOUEURS #######        
      MQTTclient.subscribe("redTeam/AntoineKia/#");

    } else {  // si echec affichage erreur
      Serial.print("ECHEC, rc=");
      Serial.print(MQTTclient.state());
      Serial.println(" nouvelle tentative dans 5 secondes");
      delay(5000);
    }
  }
}

void MQTTcallback(char* topic, byte* payload, unsigned int length) {
  Serial.print("Message MQTT reçu sur [");
  String topicString = String((char*)topic);
  Serial.print(topicString);
  Serial.println("] ");
  payload[length] = '\0';

  // ####### 2.b. "AIGUILLAGE" DES MESSAGES MQTT SELON LES TOPICS #######
  // Vous pouvez utiliser la fonction getSubTopicAtIndex(topicString, VOTRE_INDEX)
  // Par exemple getSubTopicAtIndex(topicString, 1) retournera "AntoineRcbs" 
  // pour l'entrée topicString = "redTeam/AntoineRcbs/location"
  if (getSubTopicAtIndex(topicString, 2) == "location") {
    onPlayerLocationUpdate(payload, length);
  } else if (getSubTopicAtIndex(topicString, 2) == "last-selected-construction-item") {
    onConstructionItemHeld(payload, length);
  }
}


// ####### 2.c. Décodage de la donnée (numéro du block de construction tenu) et mise à jour de l'affichage #######
// On pourra prendre un DynamicJsonDocument de taille 128
void onConstructionItemHeld(byte* payload, unsigned int length) {
  int itemId = atoi((char*)payload);
  Serial.print("Construction item hold (id) : ");
  Serial.println(itemId);
  drawMatrix(teamPixelArt, itemId);
}



// ####### 2.c. Décodage du JSON et traitement sur les données extraites #######
// On pourra prendre un DynamicJsonDocument de taille 128
void onPlayerLocationUpdate(byte* payload, unsigned int length) {
  DynamicJsonDocument playerLocation(128);
  deserializeJson(playerLocation, payload, length);
  double x, y, z, rot;
  x = playerLocation["x"];
  y = playerLocation["y"];
  z = playerLocation["z"];
  rot = playerLocation["rotation"];
  Serial.print("Distance horizontale au coffre : ");
  Serial.print(getHorizontalDistanceToRequiredChest(requiredChests, 0, x, z));
  Serial.print(", distance verticale : ");
  Serial.print(getVerticalDistanceToRequiredChest(requiredChests, 0, y));
  Serial.print(", direction :");
  double angle = getAngleDirectionDifferenceToRequiredChest(requiredChests, 0, x, z, rot);
  Serial.println(angle);
  int servoAngle = max(-90, min(90, (int) angle)) + 90;
  compassServo.write(servoAngle);  
}



void setup() { 
  initLedMatrix();
  compassServo.attach(D4, 300, 2700);
  Serial.begin(115200);

  // ####### 1.a. CONNEXION AU WIFI #######
  WiFi.begin(ssid, password);
  Serial.println("");
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("Connected to WIFI");

  // ######## 1.c. REQUETE HTTP POUR PIXEL ART ########
  fetchTeamPixelArt();  

  // ######## 1.d. REQUETE HTTP POUR COFFRES ########
  fetchRequiredChests();
   

  // ####### 2.a. INITIALISATION DU SERVEUR ET DU CALLBACK MQTT #######
   MQTTclient.setServer(mqtt_server, 1883);
   MQTTclient.setCallback(MQTTcallback);
}

void loop() {

  // ####### 2.a. RECONNEXION ET POOLING LOOP MQTT #######  
  if (!MQTTclient.connected()) {
    MQTTconnect();
  }
  MQTTclient.loop();
}